//
// Hooks for detailed progress reporting.
//
// Copyright (c) 2017 - Decisive Tactics, Inc.
//
// MIT License:
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//

#include "serial_glue.h"
#include <dispatch/dispatch.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <stdlib.h>
#include <sys/errno.h>
#include <time.h>

static int s_ack_writes = 1;

void glue_add_parent_exit_handler(void)
{
    // Exit when the parent exits
    dispatch_queue_t queue = dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0);
    dispatch_source_t source = dispatch_source_create(DISPATCH_SOURCE_TYPE_PROC, getppid(), DISPATCH_PROC_EXIT, queue);
    if (source) {
        dispatch_source_set_event_handler(source, ^{
            // The parent exited
            exit(1);
        });
        dispatch_resume(source);
    }
}

void glue_set_ack_writes(int ack)
{
    s_ack_writes = ack;
}

int glue_putchar(int in_ch)
{
    char ch = in_ch;
    ssize_t rc = glue_write(&ch, 1);
    if(rc <= 0)
    {
        return EOF;
    }
    else
    {
        return in_ch;
    }
}

int DTSafeWrite(int fd, const void * bytes, long length)
{
    static int ioSize = 0x100000;
    unsigned long towrite = length;
    unsigned long nwritten = 0;
    while(1)
    {
        long wrsize = towrite;
        if(wrsize > ioSize) wrsize = ioSize;
        long rc = write(fd, bytes, wrsize);
        if(rc == 0) break;					// eof
        if(rc < 0)
        {
            if(errno != EINTR &&
            errno != EAGAIN)
            {
                return -1;
            }
        }
        else
        {
            nwritten += rc;
            towrite -= rc;
            bytes += rc;
        }
        if(rc > wrsize) break;				// should never happen; read more than anticipated
        if(towrite == 0) break;
    }
    return nwritten;
}

long glue_write(const char * buffer, long length)
{
    ssize_t rc1 = 0;
    ssize_t rc2 = 0;
    static int s_ackChannelClosed = 0;
        
    // Write to stdout
	rc1 = DTSafeWrite(1,buffer,length);
	if(rc1 <= 0)
	{
	    exit(9);
        fprintf(stderr, "glue_write write failed\n");
        fflush(stderr);
	    return rc1;
	}
	
	// Wait for ack
	if(!s_ackChannelClosed && s_ack_writes)
	{
        unsigned char ch = 0;
        rc2 = read(4, &ch, 1);
        if(rc2 <= 0)
        {
            fprintf(stderr, "glue_write ack failed\n");
            fflush(stderr);
            s_ackChannelClosed = 1;
        }
	}

    return rc1;
}

void glue_report_cancel(void)
{
    fprintf(stderr, "%s", "\n[@CANCEL]\n");
    fflush(stderr);
}

void glue_report_filename(const char * filename)
{
    fprintf(stderr, "\n[@FNAME: %s]\n", filename);
    fflush(stderr);
}

static long s_totalBytes = -1;

void glue_report_bytes_total(long total)
{
    // Constant used for XMODEM to define unknown length (DEFBYTL)
    if(total == 2000000000)
        total = -1;
    if(total < 0) total = -1;
        
    s_totalBytes = total;
}

void glue_report_bytes_completed(long completed)
{
    fprintf(stderr, "\n[@XFER: %d/%d]\n", (int) completed, (int) s_totalBytes);
    fflush(stderr);
}


